/*:
 * @target MZ
 * @plugindesc [Add-on] HS_ReadSkipMZ：既読履歴をlocalStorageに永続化しニューゲームでも引き継ぐ v1.0.1
 * @author
 *
 * @command ClearGlobalHistory
 * @text グローバル既読履歴リセット
 * @desc localStorage上の既読履歴を削除し、必要ならリロードします。（デバッグ/検証用）
 *
 * @arg reload
 * @text 直後にリロード
 * @type boolean
 * @default true
 *
 * @arg clearGameSystem
 * @text ゲーム内既読もクリア
 * @type boolean
 * @default true
 *
 * @help
 * HS_ReadSkipMZ.js の既読履歴（$gameSystem._hsReadText）を localStorage にも保存し、
 * ニューゲームでも既読スキップが効くようにします。
 *
 * 重要：
 * - HS_ReadSkipMZ.js は NewGame 時に _hsReadText をクリアするため（設定次第）、
 *   setupNewGame の「後」でマージする必要があります。
 * - 本プラグインは HS_ReadSkipMZ.js より下に配置してください。
 *
 * - タイトル画面 隠しコマンド
 * - Ctrl + Alt + Shift + H を押すと、グローバル既読履歴（localStorage）をリセット
 * - 併せて メモリ上の辞書 と $gameSystem._hsReadText（ゲーム内既読）もクリア
 * - 画面下に「既読履歴をリセットしました」と簡易表示、OK音
 */

(() => {
  "use strict";

  const STORAGE_KEY = "HS_ReadSkipMZ_GlobalHistory";
  const SAVE_DELAY_MS = 3000;   // 既読追加→保存までの遅延
  const MAX_ENTRIES = 20000;    // 0なら無制限

  let globalHist = Object.create(null);
  let saveTimer = 0;

  function loadGlobal() {
    try {
      const raw = localStorage.getItem(STORAGE_KEY);
      if (!raw) return;
      const obj = JSON.parse(raw);
      if (obj && typeof obj === "object") globalHist = obj;
    } catch (_) {
      globalHist = Object.create(null);
    }
  }

  function scheduleSave() {
    if (saveTimer) return;
    saveTimer = setTimeout(() => {
      saveTimer = 0;
      try {
        localStorage.setItem(STORAGE_KEY, JSON.stringify(globalHist));
      } catch (_) {
        // 容量超過などは黙殺（ゲーム停止回避）
      }
    }, SAVE_DELAY_MS);
  }

  function mergeToGameSystem() {
    if (!$gameSystem) return;
    if (!$gameSystem._hsReadText) $gameSystem._hsReadText = Object.create(null);
    for (const k of Object.keys(globalHist)) {
      $gameSystem._hsReadText[k] = true;
    }
  }

  // -------------------------------------------------------------------------
  // グローバル既読履歴のクリア（localStorage + メモリ + 任意でゲーム内既読）
  // -------------------------------------------------------------------------
  const PLUGIN_NAME = "HS_ReadSkipMZ_GlobalHistory_LS";

  function clearGlobalHistoryInternal(clearGameSystem) {
    // 遅延保存が残っていると、あとで再保存され得るため潰す
    if (saveTimer) {
      clearTimeout(saveTimer);
      saveTimer = 0;
    }

    // メモリ上の辞書をクリア
    globalHist = Object.create(null);

    // localStorageも削除
    try {
      if (typeof localStorage !== "undefined") {
        localStorage.removeItem(STORAGE_KEY);
      }
    } catch (_) {
      // 失敗してもゲームを止めない
    }

    // すでにマージ済みの可能性があるため、ゲーム内既読もクリア（任意）
    if (clearGameSystem && typeof $gameSystem !== "undefined" && $gameSystem) {
      $gameSystem._hsReadText = Object.create(null);
    }
  }

  // プラグインコマンド：グローバル既読履歴リセット
  if (typeof PluginManager !== "undefined" && PluginManager.registerCommand) {
    PluginManager.registerCommand(PLUGIN_NAME, "ClearGlobalHistory", (args) => {
      const reload = String(args.reload) === "true";
      const clearGameSystem = String(args.clearGameSystem) !== "false";
      clearGlobalHistoryInternal(clearGameSystem);

      if (reload) {
        try { location.reload(); } catch (_) {}
      }
    });
  }

  // -------------------------------------------------------------------------
  // タイトル画面 隠しコマンド：Ctrl + Alt + Shift + H でグローバル既読履歴をリセット
  // ※ブラウザ/NW.js既定ショートカット（リロード等）と衝突しにくい組み合わせにしています。
  // -------------------------------------------------------------------------
  let _hsGlobalResetRequested = false;

  // 直接キーイベントを拾ってフラグを立てる（Input.keyMapperを汚さない）
  if (typeof document !== "undefined" && document.addEventListener) {
    document.addEventListener("keydown", (e) => {
      // タイトル画面でのみ受け付ける（ゲーム中に押しても後で誤爆しないように）
      const _hsIsTitle =
        typeof SceneManager !== "undefined" &&
        SceneManager._scene &&
        typeof Scene_Title !== "undefined" &&
        SceneManager._scene instanceof Scene_Title;
      if (!_hsIsTitle) return;
      const key = e.code || "";
      const isH = key === "KeyH" || e.keyCode === 72;
      if (isH && e.ctrlKey && e.altKey && e.shiftKey) {
        _hsGlobalResetRequested = true;
        // 既定の挙動がある環境でも誤動作しないよう抑止
        if (e.preventDefault) e.preventDefault();
      }
    });
  }

  function _hsIsPressed(symbol) {
    // Input.isPressedが利用できる場合はそれを優先
    try {
      if (typeof Input !== "undefined" && Input.isPressed) return Input.isPressed(symbol);
    } catch (_) {}
    return !!(typeof Input !== "undefined" && Input._currentState && Input._currentState[symbol]);
  }


  function _hsShowTitleToast(scene, text) {
    if (!scene || !Graphics) return;

    const duration = 120; // frames
    if (!scene._hsGhToast) {
      const h = 48;
      const bmp = new Bitmap(Graphics.width, h);
      const spr = new Sprite(bmp);
      spr.y = Graphics.height - h;
      spr.opacity = 0;
      scene.addChild(spr);
      scene._hsGhToast = spr;
      scene._hsGhToastFrame = 0;
    }

    const spr = scene._hsGhToast;
    spr.bitmap.clear();
    spr.bitmap.paintOpacity = 160;
    spr.bitmap.fillRect(0, 0, spr.bitmap.width, spr.bitmap.height, "#000000");
    spr.bitmap.paintOpacity = 255;
    spr.bitmap.drawText(text, 0, 0, spr.bitmap.width, spr.bitmap.height, "center");
    spr.opacity = 255;
    scene._hsGhToastFrame = duration;
  }

  // タイトル画面のupdateにフック
  if (typeof Scene_Title !== "undefined" && Scene_Title.prototype && Scene_Title.prototype.update) {
    const _Scene_Title_update = Scene_Title.prototype.update;
    Scene_Title.prototype.update = function() {
      _Scene_Title_update.call(this);

      // トーストのフェード管理
      if (this._hsGhToastFrame && this._hsGhToast) {
        this._hsGhToastFrame--;
        if (this._hsGhToastFrame <= 0) {
          this._hsGhToast.opacity = 0;
        } else if (this._hsGhToastFrame < 30) {
          this._hsGhToast.opacity = Math.floor(255 * (this._hsGhToastFrame / 30));
        }
      }

      // 隠しコマンド判定：Ctrl + Alt + Shift + H
      if (_hsGlobalResetRequested) {
        _hsGlobalResetRequested = false;
        clearGlobalHistoryInternal(true);

        // フィードバック（誤爆しても害はない程度）
        if (typeof SoundManager !== "undefined" && SoundManager.playOk) {
          SoundManager.playOk();
        }
        _hsShowTitleToast(this, "既読履歴をリセットしました");
      }
    };
  }

  function canAddNewKey(key) {
    if (!key) return false;
    if (globalHist[key]) return false;
    if (MAX_ENTRIES > 0 && Object.keys(globalHist).length >= MAX_ENTRIES) return false;
    return true;
  }

  // 初期ロード
  loadGlobal();

  // 既読登録をフック：$gameSystem と localStorage の両方へ
  if (Game_System.prototype.hsMarkTextRead) {
    const _hsMark = Game_System.prototype.hsMarkTextRead;
    Game_System.prototype.hsMarkTextRead = function(key) {
      _hsMark.call(this, key);
      if (canAddNewKey(key)) {
        globalHist[key] = true;
        scheduleSave();
      }
    };
  }

  // NewGame 完了後にマージ（★ここがv1.0.1の肝）
  const _setupNewGame = DataManager.setupNewGame;
  DataManager.setupNewGame = function() {
    _setupNewGame.call(this);
    mergeToGameSystem();
  };

  // ロード成功後にもマージ（セーブ履歴 + グローバル履歴を合成）
  if (Scene_Load && Scene_Load.prototype && Scene_Load.prototype.onLoadSuccess) {
    const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
      _onLoadSuccess.call(this);
      mergeToGameSystem();
    };
  }

})();

